//
//  NewsDetailsStoreTests.swift
//  NewsStores
//
//  Created by Michael Bernat on 03.12.2025.
//

import Foundation
import Testing
import Observation
@testable import NewsStores

@MainActor
struct NewsDetailsStoreTests {

    /// No random fails
    let newsDetailProviderMock = NewsDetailProviderMock()
    let newsDetailAlwaysFailingProvider = NewsDetailAlwaysFailingMock()
    
    func mockNewsDetail(with newsID: String) -> NewsDetail {
        NewsDetail(
            id: newsID,
            tags: [],
            date: Date.distantPast,
            title: "Title",
            perex: "Perex",
            body: "Body"
        )
    }
    
    @Test("NewsDetail status is nil when fetch not started")
    func notYetFetchedNewsDetail() async {
        let store = NewsDetailsStore(provider: newsDetailProviderMock)
        let newsID = "GivenID"
        
        #expect(store.fetchStatusByNewsID[newsID] == nil)
        #expect(store.newsDetailStatus(for: newsID) == nil)
    }
    
    @Test("NewsDetail status is in progress after fetch started")
    func notYetFetchedNewsDetailWithInitiatedFetch() async {
        let store = NewsDetailsStore(provider: newsDetailProviderMock)
        let newsID = "GivenID"
        #expect(store.fetchStatusByNewsID.isEmpty)
        
        store.startFetchIfNeeded(for: newsID)
        #expect(store.newsDetailStatus(for: newsID)?.isInProgress == true)
        
        store.stopAllNewsDetailFetches()
    }
    
    @available(iOS 26, *)
    @Test("News detail is not present in store and is fetched")
    func fetchStatusObservations1() async {
        // this test should be improved by failure on timeout
        let store = NewsDetailsStore(provider: newsDetailProviderMock)
        let newsID = "GivenID"
        
        let fetchStatusSequence = Observations { store.fetchStatusByNewsID }
        var sequenceIndex = 0
        let task = Task {
            sequenceLoop: for await fetchStatus in fetchStatusSequence {
                switch sequenceIndex {
                case 0:
                    #expect(fetchStatus[newsID]?.isFetchInProgress == true)
                case 1:
                    #expect(fetchStatus[newsID]?.isSuccess == true)
                    break sequenceLoop
                default:
                    break sequenceLoop
                }
                sequenceIndex += 1
            }
        }
        
        #expect(store.fetchStatusByNewsID.isEmpty)
        store.startFetchIfNeeded(for: newsID)
        await task.value
        #expect(sequenceIndex == 1)
        
        store.stopAllNewsDetailFetches()
    }
    
    @available(iOS 26, *)
    @Test("News detail is already available in store, not expired, immediately available")
    func fetchStatusObservations2() async {
        // this test should be improved by failure on timeout
        let store = NewsDetailsStore(provider: newsDetailProviderMock)
        let newsID = "GivenID"
        let givenNewsDetail = mockNewsDetail(with: newsID)
        let now = Date.now
        
        store.fetchStatusByNewsID[newsID] = .success(newsDetail: givenNewsDetail, timestamp: now)
        #expect(store.fetchStatusByNewsID[newsID]?.isSuccess == true)
        #expect(store.newsDetailStatus(for: newsID)?.newsDetail?.id == newsID)
        
        let fetchStatusSequence = Observations { store.fetchStatusByNewsID }
        let task = Task {
            for await fetchStatus in fetchStatusSequence {
                #expect(fetchStatus[newsID]?.isSuccess == true)
                break
            }
        }
        
        store.startFetchIfNeeded(for: newsID)
        await task.value
        
        store.stopAllNewsDetailFetches()
    }
    
    @available(iOS 26, *)
    @Test("News detail is not present in store and is fetch error")
    func fetchStatusObservations3() async {
        // this test should be improved by failure on timeout
        let store = NewsDetailsStore(provider: newsDetailAlwaysFailingProvider)
        let newsID = "GivenID"
        
        let fetchStatusSequence = Observations { store.fetchStatusByNewsID }
        var sequenceIndex = 0
        let task = Task {
            sequenceLoop: for await fetchStatus in fetchStatusSequence {
                switch sequenceIndex {
                case 0:
                    #expect(fetchStatus[newsID]?.isFetchInProgress == true)
                case 1:
                    #expect(fetchStatus[newsID]?.isError == true)
                    break sequenceLoop
                default:
                    break sequenceLoop
                }
                sequenceIndex += 1
            }
        }
        
        store.startFetchIfNeeded(for: newsID)
        await task.value
        
        store.stopAllNewsDetailFetches()
    }
}
