//
//  NewsOverviewStore.swift
//  NewsStores
//
//  Created by Michael Bernat on 02.12.2025.
//

import Observation

@MainActor
@Observable
/// News overview store
public final class NewsOverviewStore<OverviewProvider: NewsOverviewProviding>: Sendable {
    
    /// Data provider
    private let newsOverviewProvider: OverviewProvider
    
    /// Is `nil` when no `NewsOverview` is fetched so far
    public private(set) var newsOverviews: [NewsOverview]?
    
    /// When task is present, fetching from `NewsAPI` is in progress
    private(set) var fetchTask: Task<Void, Never>?
    
    /// Initializer
    public init(provider: OverviewProvider) {
        newsOverviewProvider = provider
    }
    
    /// Start FetchingNewsOverviews
    public func startUpdates() {
        guard fetchTask == nil else { return }
        fetchTask = Task.detached { [weak self] in
            // task runs in its own async context (not on main actor)
            guard let stream = self?.newsOverviewProvider.newsStream() else { return }
            for await apiOverviews in stream {
                guard let self else {
                    // task finishes
                    break
                }
                let fetchedNewsOverviews = apiOverviews.map { newsOverviewProvider.transform($0) }
                await set(newsOverviews: fetchedNewsOverviews)
            }
        }
    }
    
    /// Stop FetchingNewsOverviews
    public func stopUpdates() {
        fetchTask?.cancel()
        fetchTask = nil
    }
    
    /// Async setter
    /// - Parameter newsOverviews:
    private func set(newsOverviews: [NewsOverview]) async {
        self.newsOverviews = newsOverviews
    }
}
