//
//  NewsDetailsStore.swift
//  NewsStores
//
//  Created by Michael Bernat on 02.12.2025.
//

import Foundation
import Observation

enum NewsDetailsStoreConst {
    /// How long fetched `NewsDetail` is valid. In seconds.
    static let newsDetailSuccessValidity: TimeInterval = 300
}

@MainActor
@Observable
/// News detail store
public final class NewsDetailsStore<DetailProvider: NewsDetailProviding>: Sendable {
    
    /// Internal status
    enum NewsDetailFetchStatus: Sendable {
        case success(newsDetail: NewsDetail, timestamp: Date)
        case fetchInProgress(task: Task<Void, Never>)
        case error(NewsError)
    }
    
    /// Checks timestamp validity
    /// - Parameters:
    ///   - timestamp: date
    ///   - now: now
    /// - Returns: true or false
    static private func timestamp(_ timestamp: Date, isExpiredFor now: Date) -> Bool {
        timestamp.addingTimeInterval(NewsDetailsStoreConst.newsDetailSuccessValidity) < now
    }
    
    /// Data provider
    private let newsDetailProvider: DetailProvider
    
    /// Key is NewsDetail ID, value is NewsDetailFetchStatus
    var fetchStatusByNewsID: [String: NewsDetailFetchStatus] = [:]
    
    //MARK: -Public
    
    /// Initializer
    public init(provider: DetailProvider) {
        newsDetailProvider = provider
    }
    
    /// NewsDetailStatus for newsID
    /// - Parameter newsID: ID
    /// - Returns: NewsDetailStatus
    public func newsDetailStatus(for newsID: String) -> NewsDetailStatus? {
        guard let fetchStatus = fetchStatusByNewsID[newsID] else { return nil }
        switch fetchStatus {
        case .success(let newsDetail, _): return .newsDetail(newsDetail)
        case .fetchInProgress: return .inProgress
        case .error(let newsError): return .error(newsError)
        }
    }
    
    /// Starts news detail fetch when needed
    /// - Parameter newsID: ID
    public func startFetchIfNeeded(for newsID: String) {
        
        func makeFetchInProgress() {
            let task = Task<Void, Never> { [weak self] in
                guard let fetchStatus = await self?.fetchNewsDetail(for: newsID) else { return }
                self?.fetchStatusByNewsID[newsID] = fetchStatus
            }
            fetchStatusByNewsID[newsID] = .fetchInProgress(task: task)
        }
        
        guard let fetchStatus = fetchStatusByNewsID[newsID] else {
            // not yet fetched
            makeFetchInProgress()
            return
        }
        // fetchStatus is already available, but maybe error or expired
        switch fetchStatus {
        case let .success(_, timestamp):
            if Self.timestamp(timestamp, isExpiredFor: Date.now) {
                makeFetchInProgress()
            }
        case .fetchInProgress:
            break
        case .error:
            makeFetchInProgress()
        }
    }
    
    /// Purge all expired
    public func purgeExpiredNewsDetails() {
        purgeExpiredNewsDetails(now: Date.now)
    }
    
    /// Stop FetchingNewsOverviews
    public func stopAllNewsDetailFetches() {
        for (newsID, fetchStatus) in fetchStatusByNewsID {
            guard case let .fetchInProgress(fetchTask) = fetchStatus else { continue }
            fetchTask.cancel()
            fetchStatusByNewsID.removeValue(forKey: newsID)
        }
    }
    
    //MARK: -End of public
    
    /// Purge all expired
    /// - Parameter now: Date
    func purgeExpiredNewsDetails(now: Date) {
        for (newsID, fetchStatus) in fetchStatusByNewsID {
            guard case let .success(_, timestamp) = fetchStatus else { continue }
            if Self.timestamp(timestamp, isExpiredFor: now) {
                fetchStatusByNewsID.removeValue(forKey: newsID)
            }
        }
    }
    
    /// Concurrent fetch
    /// - Parameter newsID: ID
    /// - Returns: NewsDetailFetchStatus
    @concurrent
    nonisolated func fetchNewsDetail(for newsID: String) async -> NewsDetailFetchStatus {
        do throws(DetailProvider.APIError) {
            let apiDetail = try await newsDetailProvider.getNewsDetail(id: newsID)
            let newsDetail = newsDetailProvider.transform(apiDetail)
            // news detail has been fetched
            assert(newsID == newsDetail.id)
            return .success(newsDetail: newsDetail, timestamp: Date.now)
        } catch {
            // fetch error
            return .error(newsDetailProvider.transform(error))
        }
    }
}
