//
//  NewsAPI.swift
//  NewsReader
//
//  Created by woko on 03.11.2025.
//

import Foundation
import NewsAPIModels

public enum NewsAPI {
    static public func newsStream() -> AsyncStream<[NewsOverview]> {
        let file = Bundle.module.path(forResource: "news_overview", ofType: "json")!

        do {
            let data = try Data(contentsOf: URL(fileURLWithPath: file), options: [])
            let decoder = JSONDecoder()
            decoder.dateDecodingStrategy = .iso8601
            let newsResponse = try decoder.decode([NewsOverview].self, from: data)
            return AsyncStream { continuation in
                Task {
                    // Start with first 5 items
                    let initialCount = min(5, newsResponse.count)
                    continuation.yield(Array(newsResponse.prefix(initialCount)))

                    // Add one more item every 10 seconds
                    for i in (initialCount + 1)...newsResponse.count {
                        try? await Task.sleep(for: .seconds(10))
                        continuation.yield(Array(newsResponse.prefix(i)))
                    }

                    try? await Task.sleep(for: .seconds(1e10))
                }
            }
        } catch let error {
            assertionFailure("Got error \(error) while parsing news.")
            return AsyncStream { continuation in
                continuation.yield([])
                continuation.finish()
            }
        }
    }

    static public func getNewsDetail(_ id: String, canRandomFail: Bool = true, alwaysFails: Bool = false) async throws(NewsError) -> NewsDetail {
        if alwaysFails { throw NewsError.randomError }
        
        if canRandomFail && Int.random(in: 0..<4) == 0 {
            throw NewsError.randomError
        }

        try? await Task.sleep(for: .seconds(2))

        return .init(
            id: id,
            tags: Array(NewsTag.allCases.shuffled().prefix(2)),
            date: Date(),
            title: "Revolutionary AI System Achieves Human-Level Reasoning Capabilities",
            perex: "In a groundbreaking development that has sent shockwaves through the scientific community, researchers have unveiled an artificial intelligence system that demonstrates reasoning abilities comparable to human cognition. The system, developed over five years by an international team of scientists, represents a significant milestone in the pursuit of artificial general intelligence. Unlike previous AI models that excelled in narrow domains, this new system can adapt to unfamiliar situations and solve problems it has never encountered before. Early testing shows remarkable performance across diverse cognitive tasks including logic puzzles, creative problem-solving, and abstract reasoning. Experts suggest this breakthrough could fundamentally transform how we approach complex challenges in medicine, climate science, and beyond.",
            body: """
The artificial intelligence system, codenamed "Prometheus," has demonstrated capabilities that researchers once thought were decades away from realization. During extensive testing phases, the system successfully navigated complex ethical dilemmas, engaged in nuanced philosophical discussions, and even showed signs of what some researchers describe as creative thinking. The development team, comprising over 200 scientists from 15 countries, emphasized that this achievement resulted from a novel approach combining multiple AI architectures rather than relying on a single methodology.

What sets Prometheus apart from previous AI systems is its ability to transfer knowledge across domains seamlessly. In one striking demonstration, the system learned principles of fluid dynamics and then spontaneously applied those concepts to analyze social network patterns, drawing parallels that human experts found both surprising and insightful. This cross-domain reasoning represents a fundamental shift from the specialized AI systems that dominate current applications, which typically require extensive retraining for each new task or domain.

The ethical implications of this breakthrough have sparked intense debate within the AI research community and beyond. Leading ethicists and technologists have convened emergency sessions to discuss governance frameworks for such advanced systems. Dr. Sarah Chen, who chairs the International AI Ethics Board, stated that while the achievement is scientifically remarkable, it necessitates urgent conversations about safety protocols, transparency requirements, and the societal impact of increasingly capable AI systems. Several nations have already begun drafting legislation specifically addressing advanced AI systems with general reasoning capabilities.

From a technical perspective, Prometheus employs a revolutionary architecture that mimics aspects of human neural organization while incorporating quantum computing elements for certain processing tasks. The system utilizes over 10 trillion parameters, making it the largest and most complex AI model ever constructed. However, the development team insists that raw scale alone doesn't explain its capabilities. Instead, they attribute its success to breakthrough insights about how different types of knowledge can be represented and interconnected within artificial neural networks.

Industry leaders have responded to the announcement with a mixture of excitement and apprehension. Major technology companies are already exploring potential applications, from drug discovery and materials science to climate modeling and economic forecasting. However, some experts caution against rushing toward deployment without thorough testing and robust safety measures. The system's ability to reason about unfamiliar situations makes it potentially more unpredictable than conventional AI, raising questions about reliability and control in real-world applications.

The research team has committed to an open science approach, planning to publish their findings in peer-reviewed journals and share portions of their methodology with the broader research community. This decision, while celebrated by many scientists, has also drawn criticism from those concerned about potential misuse of such powerful technology. The team argues that transparency and collaborative oversight represent the best path forward, enabling the global scientific community to collectively address challenges and opportunities presented by this new class of AI systems.

Looking ahead, researchers predict that systems like Prometheus will become increasingly common over the next decade, fundamentally reshaping numerous industries and aspects of daily life. Educational institutions are already developing new curricula to prepare students for a world where human-AI collaboration is routine. As one team member reflected, this breakthrough represents not an ending but a beginning, opening doors to possibilities that previous generations could only imagine while simultaneously challenging us to thoughtfully navigate the profound responsibilities that accompany such powerful capabilities.
"""
        )
    }
}
